#!/bin/bash

#This script is distributed unther GPLV2 license
#Author: Abdul Pallares Calvi
#Send bugs suggestion or whatever related to the script to
#abdul@riereta.net

if ! which sed > /dev/null
then
	echo "sed not found, please install it or set PATH variable to point to it."
	exit 1
fi

if ! which snmpwalk > /dev/null
then
	echo "snmpwalk not found, please install net-snmp utils or set PATH variable to point to it."
	exit 1
fi

if ! which snmpget > /dev/null
then
	echo "snmpget not found, please install net-snmp utils or set PATH variable to point to it."
	exit 1
fi

########################some useful SNMP oids to use in code
sysDescr=".1.3.6.1.2.1.1.1.0"
sysObjectID=".1.3.6.1.2.1.1.2.0"
sysUpTimeInstance=".1.3.6.1.2.1.1.3.0"
sysContact=".1.3.6.1.2.1.1.3.0"
sysName=".1.3.6.1.2.1.1.5.0"
sysLocation=".1.3.6.1.2.1.1.6.0"
sysServices=".1.3.6.1.2.1.1.7.0"
sysORLastChange=".1.3.6.1.2.1.1.8.0"
ifInterfaces="1.3.6.1.2.1.2"
ifMib="1.3.6.1.2.1.31"

##############Variable declaration
null="> /dev/null 2&1"
snmpwalkFile=""
name=""
ip=""
options=""
device=""
output=""
oidOutputFile=""
interactive=""
list=""
includePattern=""
excludePattern=""
oidList=Array 		#Array to store selected oids to add to a template
oidDescriptions=Array	#Array to store a description for oids in oidList
oidIndexList=Array	#Array to store the index of oids in oidList

function Usage
{
	echo "Usage: $0 OPTIONS snmpwalk or snmpget options without OIDs or MIBs or device address 
	OPTIONS:
		-h  This help screen
		-a  (Mandatory) 'Address' Destination Ip Address
		-i  (Optional) 'Interactive' (aka wizard) mode

		-tf (Mandatory, except in wizard mode) 'Template File' name, the resultant template
		-om (Mandatory, except in wizard mode) 'Output Module' to use for render the template (shoule exist in template_modules directory) ie: zabbix for template_modules/zabbix.sh, nagios for template_modules/nagios.sh, * for template_modules/*.sh... 
		-if (Optional) 'Input File' with a list of oids to use, one per line in snmpwalk format
		-of (Optional) 'Output File' to the selected list of items in a snmpwalk fromat, this file can be used for -if option
		-jg (Optional) 'Joinn Graphs' makes an array of oids that should be joined, it can be a pair or more, in a graph, let it blank to use the ifInXX and ifOutXX pairs. You can add your own items to the standard ones, the format is: \"oid1+oid2,oid3+oid4+oid5+oid6,...,oidN-1+oidN\" so the plus (+) symbol split oids that will be joined and comma (,) split each oid groups, its compatible with grep -E command
		-pn (Optional) 'Print Name' prints the target system name and exit
		-pd (Optional) 'Print Description' prints the target system description and exit
		-po (Optional) 'Print Oid' prints the target value the specified OID, ex: \"$0 -a 127.0.0.1 -v1 -c public .1.3.6.1.2.1.1.6.0\" prints the sysLocation OID
		
	SEARCH OPTIONS:
		-ip (Optional) 'Include Pattern' coma separated include patterns of oid or text (Gi, fa, \.1050\ =,...) case insensitive
		-ep (Optional) 'Exclude Pattern' coma separated exclude patterns of oid or text (Vlan,Null...) case insensitive
		-gi (optional) 'Get Items' Include OID index of items that match with given patterns (Gi0/1,gi0/2) grep -Ei compatible

	NOTE: This shell variables are defined by default to use it in any template
		sysDescr=\".1.3.6.1.2.1.1.1.0\"
		sysObjectID=\".1.3.6.1.2.1.1.2.0\"
		sysUpTimeInstance=\".1.3.6.1.2.1.1.3.0\"
		sysContact=\".1.3.6.1.2.1.1.3.0\"
		sysName=\".1.3.6.1.2.1.1.5.0\"
		sysLocation=\".1.3.6.1.2.1.1.6.0\"
		sysServices=\".1.3.6.1.2.1.1.7.0\"
		sysORLastChange=\".1.3.6.1.2.1.1.8.0\"
		"
	exit 1
}

############# Argument parsing
for (( i=$BASH_ARGC-1; $i >= 0; i=$i-1 ))
do
	arg=""
	value=""
	arg=${BASH_ARGV[$i]}
	if [ $i -gt 0 ]
	then
		z=$(($i-1))
		if ! echo ${BASH_ARGV[$z]} | grep - > /dev/null 2>&1
		then
			value=${BASH_ARGV[$z]}
			((i-=1))
		fi
	fi

	case $arg in
		-h)
			Usage
		;;
		-a)
			ip=$value
			snmpwalkFile=tmp$ip
		;;
		-tf)
			templateName=$value
		;;
		-om)
			outputModule=$value
		;;
		-gi)
			getItems=$value
		;;
		-i)
			interactive="true"
		;;
		-if)
			oidInputFile=$value	

		;;
		-of)
			oidOutputFile="$value"
			if [ ! -f "$oidOutputFile" ]
			then
				echo "" > "$oidOutputFile"
			fi

			if [ ! -f "$oidOutputFile" ]
			then
				echo "file $oidOutputFile doesn't exists and im not able to create it!!!"
				exit
			fi
		;;
		-jg)
			#Oids of items that should be joined in the graphs	

			joinGraphs=( ".1.3.6.1.2.1.31.1.1.1.3.+.1.3.6.1.2.1.31.1.1.1.5." ".1.3.6.1.2.1.2.2.1.13.+.1.3.6.1.2.1.2.2.1.19." ".1.3.6.1.2.1.2.2.1.14.+.1.3.6.1.2.1.2.2.1.20." ".1.3.6.1.2.1.31.1.1.1.2.+.1.3.6.1.2.1.31.1.1.1.4." ".1.3.6.1.2.1.2.2.1.12.+.1.3.6.1.2.1.2.2.1.18." ".1.3.6.1.2.1.2.2.1.10.+.1.3.6.1.2.1.2.2.1.16." ".1.3.6.1.2.1.2.2.1.11.+.1.3.6.1.2.1.2.2.1.17."  )

			if [ -n "$value" ]
			then
				joinGraphs=( ${joinGraphs[*]} $( echo "$value" | sed -r 's/,/" "/g' | sed -r 's/^/"/' | sed -r 's/$/"/' ) )
			fi
		;;
		-ip)
			includePattern=$(echo $value | sed -E 's/,/|/g')
		;;
		-ep)
			excludePattern=$(echo $value | sed -E 's/,/|/g')
		;;
		-pn)
			printOid=$sysName
		;;
		-pd)
			printOid=$sysDescr
		;;
		-po)
			printOid=$value
		;;
		-u)
			#secname for snmpv3
			secName=$value
			options="$options $arg $value"
		;;
		-l)
			#seclevel for snmpv3
			secLevel=$value
			options="$options $arg $value"
		;;
		-A)
			#authpassphrase for snmpv3
			authPassPhrase=$value
			options="$options $arg $value"
		;;
		-X)
			#privpassphrase for snmpv3
			privPassPhrase=$value
			options="$options $arg $value"
		;;
		*)
			options="$options $arg $value"
		;;
	esac
done

if [ -z "$ip" ] 
then
	echo "IP ADDRESS IS MANDATORY (-a)"
	Usage
fi

if [ -z "$options" ]
then
	echo "SNMP OPTIONS ARE MANDATORY"
	Usage
fi

if [ "$interactive" != "true" ]
then
	if [ -z "$templateName" ]
	then
		echo "TEMPLATE NAME IS MANDATORY (-tf)"
		Usage
	fi
	if [ -z "$outputModule" ]
	then
		echo "OUTPUT TEMPLATE MODULE IS MANDATORY (-om)"
		Usage
	fi
fi

###########################  Functions declaration

function GetValues
{
	#Return the value or values of an snmpwalk or get result lines
	#It should be a grep in $snmpwalkFile to avoid network traffic
	#but for special needs
	#@Recive:
	# @$1: The lines or lines to be parsed
	echo "$( echo "$1" | sed -r 's/\ \.1\.3\.6/\n.1.3.6/g' | sed -r 's/[^"]+"([^"]*)".*/\1/' | sed -r 's/[^\(]+\(([^\)]*)\).*/\1/' | sed -r 's/[^:]+:\s*([^\("\)]+)/\1/' )"
}

function GetTypes
{
	#Return the type or types of an snmpwalk or get result lines
	#It should be a grep in $snmpwalkFile to avoid network traffic
	#but for special needs
	#@Recive:
	# @$1: The lines or lines to be parsed
	echo "$( echo "$1" | sed -r 's/\.1\.3\.6/\n.1.3.6/g' | sed -r 's/[^=]+=\s([^:]+):.*/\1/g' )"
}

function GetOids
{
	#Return the OID or OIDs of an snmpwalk or get result lines
	#It should be a grep in $snmpwalkFile to avoid network traffic
	#but for special needs
	#@Recive:
	# @$1: The lines or lines to be parsed
	echo "$( echo "$1" | sed -r 's/\.1\.3\.6/\n.1.3.6/g' | sed -r 's/([^\ =]+).*/\1/' )"
}

function GetItems
{
	#Return a list of items which match with the given regular expression
	#@Recives:
	# @$1: grep extended regular expression or any simple string
	# @$2: lines to be parsed

	echo "$( echo "$2" | grep -Ei "$1" )"
}

function ApplyFilters
{
	#Filter the given list with the user defined filters
	#@Recives
	#  @$1: The list to be parsed

	result="$1"

	if [ ! -z "$includePattern" ]
	then
		result="$( echo "$result" | grep -Ei "$includePattern" )"
	fi

	if [ ! -z "$excludePattern" ]
	then
		result="$( echo "$result" | grep -Eiv "$excludePattern" )"
	fi
	
	echo $result
	exit
	echo "$result"
}

function AddToOidTemplate
{
	#Insert a snmpwalk line into the output oid template file
	#@Recives 
	# @$1: the OID to add
	
	if ! grep -Ei "$1" $snmpwalkFile >> $oidOutputFile
	then
		return 1
	fi
	return 0
}

function ScanOid
{
	#Scan the given oid brunch allowing the user to specify
	#which of them wants to add, recursive into the brunch
	#@Recives
	# @$1: The OID brunch to be scanned

	local i=1 line="" values=Array oids=Array oidTypes=Array mibs=Array \
	choice="" oidIdx=0 add="" item="" oid=$1

	#this 2 lines allow to expand the capacity of the wizard
	#for certain types of complex items that oderwise would be
	#dirty to manage with the wizard, i.e: Network interfaces
	managed="false" #each module must to set this variable to true if success
	snmpItems=$( cat $snmpwalkFile )
	source wizard_modules/* 

	if [ $managed == "false" ]
	then
		printf "Which mibs i must to add to the $oidOutputFile template?\nComma separated list\n\n"
		oidList=$( ApplyFilters "$( GetItems "$oid\." "$snmpItems" )" )
		values=( $( GetValues $oidList ) )
		oids=( $( GetOids "$oidList" ) )
		oidTypes=( $( GetTypes "$oidList" ) )
		i=0
		for (( i=0; i<${#oids[*]}; i++)) 
		do
			mibs[$i]=$( snmptranslate ${oids[$i]} )
			echo "$i-${mibs[$i]} = ${oidTypes[$i]}: ${values[$i]}"
		done
		echo "q-Go back"
		echo -n "Make your choice: "
		read choice

		#Exit if q is pressent
		if [ "$choice" == "q" ]
		then
			clear
			return 3 
		fi

		add=( $( echo $choice | sed -r 's/,/\ /g' ) )

		for item in $add
		do
			oidIdx=$( echo ${oids[$item]} | sed -r 's/.*\.([0-9]+)$/\1/' )
			if [ "$oidIdx" -eq "0" ]
			then
				AddToOidTemplate "${oids[$item]}"
			else
				clear
				echo "The OID $item have subitems"
				ScanOid $item
			fi
		done
	fi
	return 0
}

function ScanDevice
{
	#Scan all snmp object of a device and ask the user
	#which of them wants to add to the items template
	cmd=$( echo "snmpwalk $options" | sed -r 's/-O[^\ ]*//' )
	cmd="$cmd -Ofn0 $ip"
	eval $cmd > $snmpwalkFile
	if [ ! -f "$snmpwalkFile" ]; then echo "Not snmpwalk results, revise snmp connectivity with $ip"; exit 1; fi

	clear
	echo "Scanning for management mib-2 MIBs"
	printf "\n\n"
	found=$( ( cat $snmpwalkFile | grep "1.3.6.1.2.1" | sed -r 's/.*1\.3\.6\.1\.2\.1\.([0-9]+)\..*/\1/' | sort -g | uniq ) )

	if [ -z "$found" ]; then echo "Something strange happens, run bash -x $0 to debug"; exit 1; fi

	printf "This is a list of MIBs detected at $ip, which do you want to use?\nas a comma separated list:\n\n"
	for idx in $found
	do
		grep "^\s *\<$idx\>" utils/iso.org.dod.internet.mgmt.mib-2.txt | sed -r 's/^\ *//'
	done
	echo "q-Go back"
	echo -n "Make your choice: "
	read scanOids

	#Exit if q is pressent
	if [ "$scanOids" == "q" ]
	then
		clear
		return 3
	fi

	scanOids=$( echo "$scanOids" | sed -r 's/,/\ /g' )

	for idx in $scanOids
	do
		ScanOid 1.3.6.1.2.1.$idx
	done
	return 0
}

function MenuOidFile
{
	#Manage the actions needed to build the output file for OIDs
	#items that can be used to make new output templates
	#@Reciceves
	#  @$1: new|edit, if adding or editing a file
	choice=0
	clear

	if [ -z "$oidOutputFile" ]
	then
		echo -n "Introduce an output file path: "
		read oidOutputFile
	fi

	while [ ! -f "$oidOutputFile" ]
	do
		if ! echo -n "" > "$oidOutputFile"
		then
			echo "Can't create file $oidOutputFile, error code is: $?. Try again ;)"
			echo -n "Introduce an output file path: "
			read oidOutputFile
		fi
	done

	if [ -s "$oidOutputFile" ] && [ "$1" == "new" ]
	then
		echo -n "File $oidOutputFile exists, should i remove it? y/n: "
		read remove
		if [ "$remove" == "n" ]
		then
			return 0
		fi
		rm $oidOutputFile
	fi

	clear
	while true
	do
		echo "#########OID items template creation"
		echo ""
		echo "-Editing template: '$oidOutputFile'"
		echo "-Items in template: '$( cat $oidOutputFile | wc -l )'"
		echo "-Include filter (for options 1 and 2): '$includePattern'"
		echo "-Exclude filter (for options 1 and 2): '$excludePattern'"
		echo ""
		echo "1-Scan device (interactive)"
		echo "2-Specify OID or MIB to scan (interactive)"
		echo "3-Add custom OID"
		echo "4-Remove items from the OIDs template"
		echo "5-List items in template"
		echo "6-Specify patterns (string or OID) that you want to include (grep -Ei compatible) eg: Gi,Fa"
		echo "7-Specify patterns (string or OID) that you want to exclude (grep -Eiv compatible) eg: \.10501\ =,vl"
		echo "q-Go back"
		echo -n "Make your choice: "
		read choice
		case $choice in
			1)
				ScanDevice
			;;
			2)
				oid=""
				while [ -z "$oid" ]
				do
					echo -n "Insert the OID or MIB to scan:"
					read oid
				done
				ScanOid $oid 
			;;
			3)
				oid=""
				while [ -z "$oid" ]
				do
					echo -n "Insert the OID or MIBn to scan:"
					read oid
				done

				clear
				lines=grep "$oid" $snmpwalkFile

				add="true"
				if [ "$( echo $lines | wc -l )" -eq "0" ]
				then
					printf "ERROR: The OID $oid is not found in device $ip\n\n"
					add="false"
				fi

				if [ "$( echo $lines | wc -l )" -gt "1" ]
				then
					printf "ERROR: There is more than one result,\ninsert an OID that just get one item.\nIf you want to add more than one OID select options 1,2,3 or 5"
					echo "	Items returned by this OID are: "
					echo "  $lines"
					add="false"
				fi

				if [ "$add" != "false" ]
				then
					AddToOidTemplate "$oid" "$( GetTypes $lines )" "$( GetValues $lines )"
				fi
			;;
			4)
				clear
				fileItems=( $( cat $oidOutputFile ) )
				i=0
				for item in ${fileItems[*]}
				do
					oid=$( echo $item | sed -r 's/^([^%]+).*/\1/' )
					details=$( echo $item | sed -r 's/^[^%]+(%.*)/\1/' )
					echo "$i $( snmptranslate $oid )$details"
					(( i++ ))
				done
				printf "\nWhich items do you want to remove?(Comma separated list):"
				read items
				items=( $( echo $items | sed -r 's/,/\ /g' ) )
				for item in ${items[*]}
				do
					unset fileItems[$item]
				done
				echo ${fileItems[*]} | sed -r 's/\ /\n/g' > $oidOutputFile
			;;
			5)
				clear
				fileItems=( $( cat $oidOutputFile | sed -r 's/\ /_/' ) )
				i=0
				for item in ${fileItems[*]}
				do
					oid=$( echo $item | sed -r 's/^([\_]+).*/\1/' )
					details=$( echo $item | sed -r 's/.*(=.*)/\1/' )
					echo "$i $( snmptranslate $oid )$details"
					(( i++ ))
				done
				printf "\n\n"
			;;
			6)
				clear
				echo "You can specify any grep compatible extended regular expression pattern (grep -Ei)"
				echo "If you need to use comma in the search pattern, just escape it \, ;)"
				read -r -p "What, case insensitive, patterns do you want to include (comma separated): " includePattern
				echo "If you need to use comma in the search pattern, just escape it \, ;)"
				includePattern="$( echo "$includePattern" | sed -r 's/\\,/%$%/g' | sed -r 's/,/|/g' | sed -r 's/%\$%/,/g')"
				clear
			;;
			7)
				clear
				echo "You can specify any grep compatible extended regular expression pattern (grep -Ei)"
				echo "If you need to use comma in the search pattern, just escape it \, ;)"
				read -r -p "What, case insensitive, patterns do you want to exclude (comma separated list): " excludePattern
				excludePattern="$( echo "$excludePattern" | sed -r 's/\\,/%$%/g' | sed -r 's/,/|/g' | sed -r 's/%\$%/,/g' )"
				clear
			;;
			q)
				clear
				return 3
			;;
		esac
	done
}

function MakeDeviceTemplate
{
	source "template_modules/$outputModule.sh"
}

function InteractiveSession
{
	clear
	while true
	do
		printf "Welcome to THE WIZARD, now just pray!! ;)\n\n"
		echo "Working with files:"
		if [ -z "$oidOutputFile" ]
		then
			echo "	-OID items template: You must to specify"
		else
			echo "	-OID items template: $oidOutputFile"
		fi
		echo "	-Snmpwalk temporary cache file: $snmpwalkFile"
		echo "	-Output device template: $templateName"

		echo ""
		echo "1-Build a new OID items template:"
		echo "  Creates a file with a list of items to be monitorized"
		echo "2-Edit an existing OID items template."
		echo "3-Build a device  template to use by others (zabbix, nagios, cacti...):"
		echo "  Create a file to be used in other sofwares."
		echo "  The softwares supported deppends on the modules created and added"
		echo "  to the directory template__modules"
		echo "q-Exit"
		echo -n "Make your choice: "
		read choice
		case "$choice" in
			1)
				oidOutputFile=""
				MenuOidFile "new"
			;;
			2)
				clear
				if [ -z "$oidOutputFile" ]
				then
					echo -n "Insert the path to the OIDs file: "
					read oidOutputFile
				fi

				if [ ! -f $oidOutputFile ]
				then
					echo -n "File $oidOutputFile doesn't exists, should i create it? y/n: "
					read confirm
					if [ "$confirm" == "y" ]
					then
						MenuOidFile "new"
						continue
					else
						clear
						continue
					fi
				fi
				MenuOidFile "edit"
			;;
			3)
				if [ -z "$oidOutputFile" ]
				then
					clear
					echo "Create a oid template (option 1)"
					read -p "Press any key to continue"
					clear
					continue
				fi
				oidInputFile=$oidOutputFile
				snmpItems=$( cat $oidInputFile )
				clear
				read -r -p "Insert the filename of the output template: " templateName
				clear
				i=0
				for module in $( ls template_modules/*.sh | sed -r 's/[^\/]+\/(.*)\.sh/\1/g' )
				do
					echo "$i - $module"
					modules[$i]=$module
				done
				read -r -p "Insert the template or templates to use(comma separated list): " outputModules
				for index in $( echo $outputModules | sed -r 's/,/ /g' )
				do
					outputModule=${modules[$index]}
					MakeDeviceTemplate
				done
			;;
			q)
				clear
				return 0
			;;
		esac
	done
}

function run
{
	if [ -n "$interactive" ]
	then
		if InteractiveSession
		then
			rm $snmpwalkFile > /dev/null 2>&1
			exit 0
		else
			rm $snmpwalkFile > /dev/null 2>&1
			exit 1
		fi
	fi

	if [ -n "$oidInputFile" ]
	then
		snmpItems="$( ApplyFilters "$( cat $oidInputFile )" )"
	else
		snmpItems="$( ApplyFilters "$( snmpwalk $options $ip )" )"
	fi

	if [ -n "$oidOutputFile" ] && [ -n "$snmpItems" ]
	then
		echo $snmpItems > $oidOutputFile
	fi

	if [ -n "$outputModule" ]
	then
		MakeDeviceTemplate
	fi
	exit
}

if [ ! -z "$printOid" ]
then
	echo "$( GetValues "$( snmpget $options $ip $printOid )" )"
	exit 0
fi

if [ -n "$getItems" ]
then
	if [ -z "$oidInputFile" ]
	then
		echo "No oid template file especified, its mandatory for -gi option (-if)"
		Usage
	fi

	pattern=$( echo "$getItems" | sed -r 's/,/|/g' )
	output=""
	snmpItems=$( snmpwalk $options $ip )
	output=$( GetItems "$pattern" "$snmpItems" | grep -Ei "$pattern" | sed -r 's/.*\.([0-9]+) =.*/\1/' | sort | uniq  )

	if [ -n $includePattern ]
	then
		includePattern="$includePattern|"
	fi

	includePattern=$( echo "${output[*]}" | sed -r 's/\ /|/g' )
fi

run
